using System;
using System.Data;
using System.Collections;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
//Using the broken rules of the ProlongedTransfusionReport because they are the same as
//the rules for the CostAccountingReport
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.TransfusionReport;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for TransfusionReport.
	/// </summary>
	public class TransfusionReport : VbecsReport
	{
		private ArrayList _alPhysicians;
		private ArrayList _alLocations;
		private ArrayList _alSpecialties;
		
					
		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6109"> 
		///		<ExpectedInput>Valid start and end dates</ExpectedInput>
		///		<ExpectedOutput>TransfusionReport object created</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6110"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		public TransfusionReport(DateTime startDate, DateTime endDate):this()
		{
			this.StartDate = startDate;
			this.EndDate = endDate;		
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6111"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>TransfusionReport object created</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6112"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Blank Constructor
		/// </summary>
		public TransfusionReport()
		{
			SetInitialBrokenRules( BROKEN.FormClassName, BROKEN.RuleSets.BaseRuleSet );
			
			this._alPhysicians=new ArrayList();
			this._alLocations=new ArrayList();
			this._alSpecialties=new ArrayList();

			this.IssuingPhysicianReport=true;
			this.TreatingSpecialityReport=false;
			this.IssuedToLocationReport=false;

			this.IsScheduled=false;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6171"> 
		///		<ExpectedInput>Valid physician name</ExpectedInput>
		///		<ExpectedOutput>Name added to physician array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6172"> 
		///		<ExpectedInput>Invalid physician name (whitespace only)</ExpectedInput>
		///		<ExpectedOutput>Name not added to phsyician array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AddPhysician
		/// </summary>
		/// <param name="physician"></param>
		public void AddPhysician(string physician)
		{
			if(!this._alPhysicians.Contains(physician))
			{
				if(physician.Trim().Length>0 && this.IssuingPhysicianReport)
				{
					this._alPhysicians.Add(physician);
				}	
				this.RuleBroken(BROKEN.Physicians, this.IssuingPhysicianReport && this._alPhysicians.Count < 1 ); 
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6173"> 
		///		<ExpectedInput>Valid name part of physician array</ExpectedInput>
		///		<ExpectedOutput>Name removed from physician array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6174"> 
		///		<ExpectedInput>Invalid name not part of physician array</ExpectedInput>
		///		<ExpectedOutput>Name not removed from physician array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RemovePhysician
		/// </summary>
		/// <param name="physician"></param>
		public void RemovePhysician(string physician)
		{
			if(this._alPhysicians.Contains(physician))
			{
				this._alPhysicians.Remove(physician);
			}
			this.RuleBroken(BROKEN.Physicians, this.IssuingPhysicianReport && this._alPhysicians.Count < 1 ); 
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6175"> 
		///		<ExpectedInput>Valid location name</ExpectedInput>
		///		<ExpectedOutput>Name added to location array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6176"> 
		///		<ExpectedInput>Invalid location name (whitespace only)</ExpectedInput>
		///		<ExpectedOutput>Name not added to location array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AddLocation
		/// </summary>
		/// <param name="location"></param>
		public void AddLocation(string location)
		{
			if(!this._alLocations.Contains(location))
			{
				if(location.Trim().Length>0 && this.IssuedToLocationReport)
				{
					this._alLocations.Add(location);
				}
			}
			this.RuleBroken(BROKEN.Locations, this.IssuedToLocationReport && this._alLocations.Count < 1); 
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6177"> 
		///		<ExpectedInput>Valid location part of location array</ExpectedInput>
		///		<ExpectedOutput>Location removed from location array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6178"> 
		///		<ExpectedInput>Invalid location not part of location array</ExpectedInput>
		///		<ExpectedOutput>Location not removed from location array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RemoveLocation
		/// </summary>
		/// <param name="location"></param>
		public void RemoveLocation(string location)
		{
			if(this._alLocations.Contains(location))
			{
				this._alLocations.Remove(location);
			}
			this.RuleBroken(BROKEN.Locations, this.IssuedToLocationReport && this._alLocations.Count < 1); 
			
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6179"> 
		///		<ExpectedInput>Valid treating specialty</ExpectedInput>
		///		<ExpectedOutput>Treating specialty added to treating specialty array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6180"> 
		///		<ExpectedInput>Invalid treating specialty (whitespace only)</ExpectedInput>
		///		<ExpectedOutput>Treating specialty not added to treating specialty array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// AddSpecialty
		/// </summary>
		/// <param name="specialty"></param>
		public void AddSpecialty(string specialty)
		{
			if(!this._alSpecialties.Contains(specialty))
			{
				if(specialty.Trim().Length>0 && this.TreatingSpecialityReport)
				{
					this._alSpecialties.Add(specialty);
				}
				this.RuleBroken(BROKEN.Specialty, this.TreatingSpecialityReport && this._alSpecialties.Count < 1); 
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6181"> 
		///		<ExpectedInput>Valid treating specialty part of treating specialty array</ExpectedInput>
		///		<ExpectedOutput>Treating specialty removed from treating specialty array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6182"> 
		///		<ExpectedInput>Invalid treating specialty not part of treating specialty array</ExpectedInput>
		///		<ExpectedOutput>Treating specialty not removed from treating specialty array</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// RemoveSpecialty
		/// </summary>
		/// <param name="specialty"></param>
		public void RemoveSpecialty(string specialty)
		{
			if(this._alSpecialties.Contains(specialty))
			{
				this._alSpecialties.Remove(specialty);
			}
			this.RuleBroken(BROKEN.Specialty, this.TreatingSpecialityReport && this._alSpecialties.Count < 1); 
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6183"> 
		///		<ExpectedInput>Valid physician, location and treating specialty arrays</ExpectedInput>
		///		<ExpectedOutput>Filter statement generated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6184"> 
		///		<ExpectedInput>No report type set</ExpectedInput>
		///		<ExpectedOutput>No filter statement generated</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GenerateFilterStatement
		/// </summary>
		/// <returns></returns>
		public string GenerateFilterStatement()
		{
			if(this.IsValid)
			{
				System.Text.StringBuilder searchString = new System.Text.StringBuilder();

				// The following if statements are determining
				// what report is being generated and creating the 
				// appropriate search where clause.
				if (this.IssuingPhysicianReport)
				{
					for(int i=0; i<this._alPhysicians.Count; i++)
					{	
						searchString.Append("|" + this._alPhysicians[i].ToString());
					}
					searchString.Append("|");
				}

				if (this.IssuedToLocationReport)
				{
					for(int i=0; i<this._alLocations.Count; i++)
					{
						searchString.Append("|" + this._alLocations[i].ToString());
					}
					searchString.Append("|");
				}

				if (this.TreatingSpecialityReport)
				{
					for(int i=0; i<this._alSpecialties.Count; i++)
					{
						searchString.Append("|" + this._alSpecialties[i].ToString());
					}
					searchString.Append("|");
				}

				// Return the string that will be then be passed
				// to the stored procedure.
				return searchString.ToString();
			}
			else
			{
				return String.Empty;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6113"> 
		///		<ExpectedInput>Boolean value</ExpectedInput>
		///		<ExpectedOutput>IssuedToLocationReport property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6114"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IssuedToLocationReport
		/// </summary>
		public bool IssuedToLocationReport
		{
			get
			{
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IssuedToLocationReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IssuedToLocationReport]=value;
				this.RuleBroken(BROKEN.Locations, this.IssuedToLocationReport && this._alLocations.Count < 1); 
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6115"> 
		///		<ExpectedInput>Boolean value</ExpectedInput>
		///		<ExpectedOutput>TreatingSpecialityReport property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6116"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// TreatingSpecialityReport
		/// </summary>
		public bool TreatingSpecialityReport
		{
			get
			{
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.TreatingSpecialityReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.TreatingSpecialityReport]=value;
				this.RuleBroken(BROKEN.Specialty, this.TreatingSpecialityReport && this._alSpecialties.Count < 1); 
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6117"> 
		///		<ExpectedInput>Boolean value</ExpectedInput>
		///		<ExpectedOutput>IssuingPhysicianReport property updated</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6118"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IssuingPhysicianReport
		/// </summary>
		public bool IssuingPhysicianReport
		{
			get
			{
				return Convert.ToBoolean(this.ParameterRow[ARTIFICIAL.IssuingPhysicianReport]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.IssuingPhysicianReport]=value;
				this.RuleBroken(BROKEN.Physicians, this.IssuingPhysicianReport && this._alPhysicians.Count < 1 ); 
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6107"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>FilterStatement property set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6108"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// FilterStatement
		/// </summary>
		public string FilterStatement
		{
			get
			{
				return Convert.ToString(this.ParameterRow[ARTIFICIAL.FilterStatement]);
			}
			set
			{
				this.ParameterRow[ARTIFICIAL.FilterStatement]=value;
			}
		}
				
        ///<Developer>Rob Heiberger</Developer>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6121"> 
		///		<ExpectedInput>Valid TransfusionReport object with IssuedToLocationReport property of true</ExpectedInput>
		///		<ExpectedOutput>TransfusionReport dataset returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6122"> 
		///		<ExpectedInput>Invalid DivisionCode property</ExpectedInput>
		///		<ExpectedOutput>Empty data set returned</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GenerateTransfusionReportSource
		/// </summary>
		/// <returns></returns>
		public DataSet GenerateTransfusionReportSource()
		{
			DataTable dtTransfusionReport = DAL.PatientTransfusion.GetTransfusionReport(
				this.StartDate, this.EndDate, this.FilterStatement, this.DivisionCode, 
				TranslateReportTypeToParameter());
			dtTransfusionReport.Columns.Add(ARTIFICIAL.PGuid);
			dtTransfusionReport.Columns.Add(ARTIFICIAL.TransfusionType);
			dtTransfusionReport.Columns.Add(TABLE.ExceptionPatientTest.PatientName);

			for (int i=dtTransfusionReport.Rows.Count-1; i>=0; i--)
			{
				dtTransfusionReport.Rows[i][ARTIFICIAL.PGuid] = dtTransfusionReport.Rows[i][TABLE.Patient.PatientGuid].ToString();
				if (i < dtTransfusionReport.Rows.Count-1)
				{
					if (dtTransfusionReport.Rows[i][TABLE.PatientTransfusion.PatientTransfusionGuid].ToString() ==
						dtTransfusionReport.Rows[i+1][TABLE.PatientTransfusion.PatientTransfusionGuid].ToString())
					{
						dtTransfusionReport.Rows[i][TABLE.PatientTransfusionComment.PatientTransfusionCommentText] +=
							dtTransfusionReport.Rows[i+1][TABLE.PatientTransfusionComment.PatientTransfusionCommentText].ToString();
						dtTransfusionReport.Rows.RemoveAt(i+1);
					}
				}

				if (dtTransfusionReport.Rows[i].IsNull(TABLE.BloodUnitStatus.PresumedTransfusedIndicator) == false)
				{
					if ((bool)dtTransfusionReport.Rows[i][TABLE.BloodUnitStatus.PresumedTransfusedIndicator] == true)
					{
						dtTransfusionReport.Rows[i][ARTIFICIAL.TransfusionType] = "P";
					}
					else
					{
						dtTransfusionReport.Rows[i][ARTIFICIAL.TransfusionType] = "T";
					}
				}
				else
				{
					dtTransfusionReport.Rows[i][ARTIFICIAL.TransfusionType] = "T";
				}
				// CR 1947: adding patient's middle name (if applicable)
				string patientMiddleName = string.Empty;
				//
				if ( !dtTransfusionReport.Rows[i].IsNull(TABLE.Patient.PatientMiddleName) )
				{
					patientMiddleName = string.Concat( " ", dtTransfusionReport.Rows[i][TABLE.Patient.PatientMiddleName].ToString() );
				}
				//
				dtTransfusionReport.Rows[i][TABLE.ExceptionPatientTest.PatientName] = 
					string.Concat(	dtTransfusionReport.Rows[i][TABLE.Patient.PatientLastName].ToString(), 
									", ",
									dtTransfusionReport.Rows[i][TABLE.Patient.PatientFirstName].ToString(),
									patientMiddleName );
			}

			return dtTransfusionReport.DataSet;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/15/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7887"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7888"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate and print report
		/// Used by VbecsConsoleReport application
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            string whereClause = "";
            string reportCriteria = "";

            whereClause = this.GenerateFilterStatement();

            if (this.TreatingSpecialityReport)
                reportCriteria = "Treating Specialties: ";
            else if (this.IssuingPhysicianReport)
                reportCriteria = "Issuing Physicians: ";
            else if (this.IssuedToLocationReport)
                reportCriteria = "Issued To Locations: ";

            reportCriteria += whereClause.Replace("|", ",").Substring(1, whereClause.Length - 2);

            this.FilterStatement = whereClause;
            
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            DataSet reportSource = this.GenerateTransfusionReportSource();
            string dateCriteria = "Date Range: ";
            dateCriteria += Common.VBECSDateTime.FormatDateString(this.StartDate);
            dateCriteria += " - ";
            dateCriteria += Common.VBECSDateTime.FormatDateString(this.EndDate);

            reportCriteria = dateCriteria + ", " + reportCriteria;
            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(reportCriteria);

            ReportDataSource reportSourceMain = new ReportDataSource();
            if (reportSource.Tables.Count > 0)
            {
                reportSourceMain.Name = "DivisionTransfusionDataSet";
                reportSourceMain.Value = reportSource.Tables[0];
            }

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            LocalReport report = new LocalReport();

            if (this.IssuingPhysicianReport)
            {
                Stream streamPhysician = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.DivisionTransfusionReport.DivisionTransfusionByPhysician.rdlc");
                report.DataSources.Add(reportSourceHeader);
                report.DataSources.Add(reportSourceFooter);
                report.DataSources.Add(reportSourceMain);
                report.LoadReportDefinition(streamPhysician);
            }
            if (this.IssuedToLocationReport)
            {
                Stream streamIssueToLocation = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.DivisionTransfusionReport.DivisionTransfusionByIssueToLocation.rdlc");
                report.DataSources.Add(reportSourceHeader);
                report.DataSources.Add(reportSourceFooter);
                report.DataSources.Add(reportSourceMain);
                report.LoadReportDefinition(streamIssueToLocation);
            }
            if (this.TreatingSpecialityReport)
            {
                Stream streamTreatingSpecialty = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.DivisionTransfusionReport.DivisionTransfusionByTreatingSpecialty.rdlc");
                report.DataSources.Add(reportSourceHeader);
                report.DataSources.Add(reportSourceFooter);
                report.DataSources.Add(reportSourceMain);
                report.LoadReportDefinition(streamTreatingSpecialty);
            }
            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
        }

		//Helper method used to pass the appropriate report type value to the stored procedure
		private int TranslateReportTypeToParameter()
		{
			if (this.IssuingPhysicianReport)
			{
				return 1;
			}
			if (this.IssuedToLocationReport)
			{
				return 2;
			}
			if (this.TreatingSpecialityReport)
			{
				return 3;
			}
			return 0;
		}
	}
}
